<?php
/**
 * Classe de gestion du Rate Limiting
 * 
 * @package GMB_Article_Generateur
 * @since 1.0.0
 */

if (!defined('ABSPATH')) {
    exit;
}

class GMB_AG_Rate_Limiter {

    private static $transient_prefix = 'gmb_ag_rate_';
    private static $window_duration = 3600;

    public static function check($request) {
        $rate_limit = (int) get_option('gmb_ag_rate_limit', 60);
        if ($rate_limit <= 0) {
            return true;
        }

        $ip = GMB_AG_Auth::get_client_ip();
        $transient_key = self::$transient_prefix . md5($ip);
        $data = get_transient($transient_key);

        if (false === $data) {
            set_transient($transient_key, array('count' => 1, 'first_request' => time()), self::$window_duration);
            return true;
        }

        if ($data['count'] >= $rate_limit) {
            $reset_time = $data['first_request'] + self::$window_duration;
            $retry_after = $reset_time - time();

            GMB_AG_Logger::log(array(
                'request_type' => $request->get_method(),
                'endpoint'     => $request->get_route(),
                'status'       => 'rate_limited',
                'message'      => sprintf(__('Rate limit atteint (%d/%d)', 'gmb-article-generateur'), $data['count'], $rate_limit),
            ));

            return new WP_Error('gmb_rate_limit_exceeded', sprintf(__('Limite atteinte (%d/h). Reessayez dans %d min.', 'gmb-article-generateur'), $rate_limit, ceil($retry_after / 60)), array('status' => 429, 'retry_after' => $retry_after));
        }

        $data['count']++;
        set_transient($transient_key, $data, self::$window_duration);
        return true;
    }

    public static function get_info($ip = null) {
        if (null === $ip) {
            $ip = GMB_AG_Auth::get_client_ip();
        }
        $rate_limit = (int) get_option('gmb_ag_rate_limit', 60);
        $data = get_transient(self::$transient_prefix . md5($ip));

        if (false === $data) {
            return array('limit' => $rate_limit, 'remaining' => $rate_limit, 'reset' => time() + self::$window_duration, 'used' => 0);
        }

        return array(
            'limit'     => $rate_limit,
            'remaining' => max(0, $rate_limit - $data['count']),
            'reset'     => $data['first_request'] + self::$window_duration,
            'used'      => $data['count'],
        );
    }

    public static function add_headers($response) {
        $info = self::get_info();
        $response->header('X-RateLimit-Limit', $info['limit']);
        $response->header('X-RateLimit-Remaining', $info['remaining']);
        $response->header('X-RateLimit-Reset', $info['reset']);
        return $response;
    }
}
