<?php
/**
 * Classe de gestion des medias
 * 
 * @package GMB_Article_Generateur
 * @since 1.0.0
 */

if (!defined('ABSPATH')) {
    exit;
}

class GMB_AG_Media {

    private static $allowed_types = array('image/jpeg' => 'jpg', 'image/png' => 'png', 'image/gif' => 'gif', 'image/webp' => 'webp');
    private static $max_file_size = 5242880;

    public static function upload_from_url($url, $post_id, $set_featured = true) {
        if (!filter_var($url, FILTER_VALIDATE_URL)) {
            return new WP_Error('gmb_invalid_image_url', __('URL invalide.', 'gmb-article-generateur'));
        }

        require_once ABSPATH . 'wp-admin/includes/file.php';
        require_once ABSPATH . 'wp-admin/includes/media.php';
        require_once ABSPATH . 'wp-admin/includes/image.php';

        $response = wp_remote_head($url, array('timeout' => 10, 'sslverify' => false));
        if (is_wp_error($response)) {
            return new WP_Error('gmb_image_head_failed', $response->get_error_message());
        }

        $content_type = wp_remote_retrieve_header($response, 'content-type');
        $content_type = preg_replace('/;.*/', '', $content_type);
        
        if (!array_key_exists($content_type, self::$allowed_types)) {
            return new WP_Error('gmb_invalid_image_type', __('Type non autorise. Acceptes: JPG, PNG, GIF, WebP.', 'gmb-article-generateur'));
        }

        $temp_file = download_url($url, 30);
        if (is_wp_error($temp_file)) {
            return new WP_Error('gmb_image_download_failed', $temp_file->get_error_message());
        }

        $filename = 'gmb-image-' . time() . '.' . self::$allowed_types[$content_type];
        $file_array = array('name' => $filename, 'tmp_name' => $temp_file, 'error' => 0, 'size' => filesize($temp_file));

        $attachment_id = media_handle_sideload($file_array, $post_id);

        if (file_exists($temp_file)) {
            @unlink($temp_file);
        }

        if (is_wp_error($attachment_id)) {
            return new WP_Error('gmb_image_upload_failed', $attachment_id->get_error_message());
        }

        if ($set_featured) {
            set_post_thumbnail($post_id, $attachment_id);
        }

        update_post_meta($attachment_id, '_gmb_ag_source_url', esc_url_raw($url));
        return $attachment_id;
    }

    public static function find_existing_by_url($url) {
        global $wpdb;
        $attachment_id = $wpdb->get_var($wpdb->prepare(
            "SELECT post_id FROM {$wpdb->postmeta} WHERE meta_key = '_gmb_ag_source_url' AND meta_value = %s LIMIT 1",
            esc_url_raw($url)
        ));
        return $attachment_id ? (int) $attachment_id : false;
    }

    public static function upload_or_reuse($url, $post_id, $set_featured = true) {
        $existing_id = self::find_existing_by_url($url);
        if ($existing_id) {
            if ($set_featured) {
                set_post_thumbnail($post_id, $existing_id);
            }
            return $existing_id;
        }
        return self::upload_from_url($url, $post_id, $set_featured);
    }
}
