<?php
/**
 * Classe de gestion des logs
 * 
 * @package GMB_Article_Generateur
 * @since 1.0.0
 */

if (!defined('ABSPATH')) {
    exit;
}

class GMB_AG_Logger {

    private static $table_name = null;

    private static function get_table_name() {
        if (null === self::$table_name) {
            global $wpdb;
            self::$table_name = $wpdb->prefix . 'gmb_ag_logs';
        }
        return self::$table_name;
    }

    public static function log($data) {
        if (!get_option('gmb_ag_enable_logging', 1)) {
            return false;
        }

        global $wpdb;

        $defaults = array(
            'date_created' => current_time('mysql'),
            'request_type' => 'POST',
            'endpoint'     => '',
            'ip_address'   => self::get_client_ip(),
            'post_id'      => null,
            'post_title'   => null,
            'status'       => 'success',
            'message'      => '',
            'request_data' => null,
        );

        $data = wp_parse_args($data, $defaults);

        if (is_array($data['request_data'])) {
            $data['request_data'] = wp_json_encode($data['request_data'], JSON_UNESCAPED_UNICODE);
        }

        $result = $wpdb->insert(
            self::get_table_name(),
            array(
                'date_created' => $data['date_created'],
                'request_type' => substr($data['request_type'], 0, 10),
                'endpoint'     => substr($data['endpoint'], 0, 100),
                'ip_address'   => substr($data['ip_address'], 0, 45),
                'post_id'      => $data['post_id'],
                'post_title'   => $data['post_title'] ? substr($data['post_title'], 0, 255) : null,
                'status'       => substr($data['status'], 0, 20),
                'message'      => $data['message'],
                'request_data' => $data['request_data'],
            ),
            array('%s', '%s', '%s', '%s', '%d', '%s', '%s', '%s', '%s')
        );

        return $result ? $wpdb->insert_id : false;
    }

    public static function get_client_ip() {
        $ip_keys = array('HTTP_CF_CONNECTING_IP', 'HTTP_X_FORWARDED_FOR', 'HTTP_X_REAL_IP', 'HTTP_CLIENT_IP', 'REMOTE_ADDR');
        foreach ($ip_keys as $key) {
            if (!empty($_SERVER[$key])) {
                $ip = $_SERVER[$key];
                if (strpos($ip, ',') !== false) {
                    $ip = trim(explode(',', $ip)[0]);
                }
                if (filter_var($ip, FILTER_VALIDATE_IP)) {
                    return $ip;
                }
            }
        }
        return '0.0.0.0';
    }

    public static function get_logs($args = array()) {
        global $wpdb;

        $defaults = array(
            'per_page' => 20,
            'page'     => 1,
            'status'   => '',
            'search'   => '',
            'orderby'  => 'date_created',
            'order'    => 'DESC',
        );

        $args = wp_parse_args($args, $defaults);
        $table = self::get_table_name();
        $where = array('1=1');
        $values = array();

        if (!empty($args['status'])) {
            $where[] = 'status = %s';
            $values[] = $args['status'];
        }

        if (!empty($args['search'])) {
            $where[] = '(post_title LIKE %s OR ip_address LIKE %s OR message LIKE %s)';
            $search_term = '%' . $wpdb->esc_like($args['search']) . '%';
            $values[] = $search_term;
            $values[] = $search_term;
            $values[] = $search_term;
        }

        $where_clause = implode(' AND ', $where);
        $orderby = in_array($args['orderby'], array('id', 'date_created', 'status', 'ip_address', 'post_title')) ? $args['orderby'] : 'date_created';
        $order = strtoupper($args['order']) === 'ASC' ? 'ASC' : 'DESC';
        $per_page = absint($args['per_page']);
        $offset = (absint($args['page']) - 1) * $per_page;

        $query = "SELECT * FROM $table WHERE $where_clause ORDER BY $orderby $order LIMIT %d OFFSET %d";
        $values[] = $per_page;
        $values[] = $offset;

        $logs = $wpdb->get_results($wpdb->prepare($query, $values), ARRAY_A);

        $count_query = "SELECT COUNT(*) FROM $table WHERE $where_clause";
        $total = !empty($values) && count($values) > 2 
            ? $wpdb->get_var($wpdb->prepare($count_query, array_slice($values, 0, -2)))
            : $wpdb->get_var($count_query);

        return array(
            'logs'     => $logs ? $logs : array(),
            'total'    => (int) $total,
            'pages'    => ceil($total / $per_page),
            'current'  => absint($args['page']),
            'per_page' => $per_page,
        );
    }

    public static function clear_all() {
        global $wpdb;
        return $wpdb->query("TRUNCATE TABLE " . self::get_table_name());
    }

    public static function get_stats() {
        global $wpdb;
        $table = self::get_table_name();

        return array(
            'total'      => (int) $wpdb->get_var("SELECT COUNT(*) FROM $table"),
            'last_24h'   => (int) $wpdb->get_var($wpdb->prepare("SELECT COUNT(*) FROM $table WHERE date_created > %s", date('Y-m-d H:i:s', strtotime('-24 hours')))),
            'unique_ips' => (int) $wpdb->get_var("SELECT COUNT(DISTINCT ip_address) FROM $table"),
        );
    }

    public static function export_csv() {
        $result = self::get_logs(array('per_page' => 10000, 'page' => 1));
        $output = fopen('php://temp', 'r+');
        fputcsv($output, array('ID', 'Date', 'Type', 'Endpoint', 'IP', 'Post ID', 'Titre', 'Statut', 'Message'));
        foreach ($result['logs'] as $log) {
            fputcsv($output, array($log['id'], $log['date_created'], $log['request_type'], $log['endpoint'], $log['ip_address'], $log['post_id'], $log['post_title'], $log['status'], $log['message']));
        }
        rewind($output);
        $csv = stream_get_contents($output);
        fclose($output);
        return $csv;
    }
}
