<?php
/**
 * Classe client pour le systeme de licence GMB Club
 * 
 * @package GMB_Article_Generateur
 * @since 1.1.0
 */

if (!defined('ABSPATH')) {
    exit;
}

class GMB_AG_License {

    private static $api_url = 'https://license.a2z-automation.com/api/license';
    private static $secret_key = '2RiSsJc01FPsI0tuZGjabrMv479VMu3u';
    
    /**
     * Genere la signature HMAC
     */
    private static function generate_signature($license_key, $site_url, $timestamp) {
        $data = $license_key . $site_url . $timestamp;
        return hash_hmac('sha256', $data, self::$secret_key);
    }
    
    /**
     * Envoie une requete a l'API de licence
     */
    private static function api_request($endpoint, $data) {
        $license_key = get_option('gmb_ag_license_key', '');
        $site_url = home_url();
        $timestamp = time();
        
        if (empty($license_key) && $endpoint !== 'check') {
            return array('valid' => false, 'message' => 'Cle de licence manquante');
        }
        
        $data['license_key'] = $license_key;
        $data['site_url'] = $site_url;
        
        $signature = self::generate_signature($license_key, $site_url, $timestamp);
        
        $response = wp_remote_post(self::$api_url . '/' . $endpoint, array(
            'timeout' => 15,
            'headers' => array(
                'Content-Type' => 'application/json',
                'X-Signature' => $signature,
                'X-Timestamp' => $timestamp,
            ),
            'body' => wp_json_encode($data),
            'sslverify' => true,
        ));
        
        if (is_wp_error($response)) {
            return array(
                'valid' => false,
                'error' => true,
                'message' => $response->get_error_message()
            );
        }
        
        $body = wp_remote_retrieve_body($response);
        $result = json_decode($body, true);
        
        if (json_last_error() !== JSON_ERROR_NONE) {
            return array(
                'valid' => false,
                'error' => true,
                'message' => 'Reponse invalide du serveur'
            );
        }
        
        return $result;
    }
    
    /**
     * Active la licence sur ce site
     */
    public static function activate($license_key = null) {
        if ($license_key) {
            update_option('gmb_ag_license_key', sanitize_text_field($license_key));
        }
        
        $data = array(
            'site_name' => get_bloginfo('name'),
            'admin_email' => get_option('admin_email'),
            'wp_version' => get_bloginfo('version'),
            'php_version' => PHP_VERSION,
            'plugin_version' => GMB_AG_VERSION,
        );
        
        $result = self::api_request('activate', $data);
        
        if (!empty($result['valid'])) {
            update_option('gmb_ag_license_status', 'active');
            update_option('gmb_ag_license_type', $result['license_type'] ?? 'basic');
            update_option('gmb_ag_license_expires', $result['expires_at'] ?? '');
            update_option('gmb_ag_license_last_check', time());
            
            // Planifier le heartbeat quotidien
            if (!wp_next_scheduled('gmb_ag_license_heartbeat')) {
                wp_schedule_event(time() + 3600, 'daily', 'gmb_ag_license_heartbeat');
            }
        } else {
            update_option('gmb_ag_license_status', 'invalid');
        }
        
        return $result;
    }
    
    /**
     * Verifie si la licence est valide
     */
    public static function check() {
        $result = self::api_request('check', array());
        
        if (!empty($result['valid'])) {
            update_option('gmb_ag_license_status', $result['status'] ?? 'active');
            update_option('gmb_ag_license_type', $result['license_type'] ?? 'basic');
            update_option('gmb_ag_license_expires', $result['expires_at'] ?? '');
            update_option('gmb_ag_license_last_check', time());
        } else {
            update_option('gmb_ag_license_status', $result['status'] ?? 'invalid');
        }
        
        return $result;
    }
    
    /**
     * Envoie un heartbeat avec les stats
     */
    public static function heartbeat() {
        $stats = self::get_usage_stats();
        
        $result = self::api_request('heartbeat', array(
            'stats' => $stats,
            'plugin_version' => GMB_AG_VERSION,
        ));
        
        if (!empty($result['valid'])) {
            update_option('gmb_ag_license_status', 'active');
            update_option('gmb_ag_license_last_check', time());
            // Reset les stats du jour
            update_option('gmb_ag_stats_today', array(
                'articles_created' => 0,
                'api_calls' => 0,
                'api_errors' => 0,
                'date' => date('Y-m-d'),
            ));
        }
        
        return $result;
    }
    
    /**
     * Desactive la licence sur ce site
     */
    public static function deactivate() {
        $result = self::api_request('deactivate', array());
        
        // Nettoyer les options
        delete_option('gmb_ag_license_status');
        delete_option('gmb_ag_license_type');
        delete_option('gmb_ag_license_expires');
        delete_option('gmb_ag_license_last_check');
        
        // Supprimer le cron
        wp_clear_scheduled_hook('gmb_ag_license_heartbeat');
        
        return $result;
    }
    
    /**
     * Recupere les stats d'utilisation
     */
    private static function get_usage_stats() {
        $stats = get_option('gmb_ag_stats_today', array(
            'articles_created' => 0,
            'api_calls' => 0,
            'api_errors' => 0,
            'date' => date('Y-m-d'),
        ));
        
        // Si c'est un nouveau jour, reset
        if (isset($stats['date']) && $stats['date'] !== date('Y-m-d')) {
            $stats = array(
                'articles_created' => 0,
                'api_calls' => 0,
                'api_errors' => 0,
                'date' => date('Y-m-d'),
            );
        }
        
        return $stats;
    }
    
    /**
     * Incremente un compteur de stats
     */
    public static function increment_stat($stat_name) {
        $stats = get_option('gmb_ag_stats_today', array(
            'articles_created' => 0,
            'api_calls' => 0,
            'api_errors' => 0,
            'date' => date('Y-m-d'),
        ));
        
        // Si c'est un nouveau jour, reset
        if (isset($stats['date']) && $stats['date'] !== date('Y-m-d')) {
            $stats = array(
                'articles_created' => 0,
                'api_calls' => 0,
                'api_errors' => 0,
                'date' => date('Y-m-d'),
            );
        }
        
        if (isset($stats[$stat_name])) {
            $stats[$stat_name]++;
        }
        
        update_option('gmb_ag_stats_today', $stats);
    }
    
    /**
     * Verifie si le plugin peut fonctionner
     */
    public static function is_valid() {
        $status = get_option('gmb_ag_license_status', '');
        $license_key = get_option('gmb_ag_license_key', '');
        
        // Si pas de licence ou pas de statut → BLOQUER
        if (empty($license_key) || empty($status)) {
            return false;
        }
        
        // Verifier le statut
        if ($status === 'active' || $status === 'trial') {
            // Verifier si on a check recemment (cache de 12h)
            $last_check = get_option('gmb_ag_license_last_check', 0);
            if (time() - $last_check > 43200) {
                // Faire un check en arriere-plan
                self::check();
            }
            return true;
        }
        
        return false;
    }
    
    /**
     * Retourne le statut actuel de la licence
     */
    public static function get_status() {
        return array(
            'license_key' => get_option('gmb_ag_license_key', ''),
            'status' => get_option('gmb_ag_license_status', 'unknown'),
            'type' => get_option('gmb_ag_license_type', ''),
            'expires_at' => get_option('gmb_ag_license_expires', ''),
            'last_check' => get_option('gmb_ag_license_last_check', 0),
        );
    }
    
    /**
     * Message d'erreur si licence invalide
     */
    public static function get_blocked_message() {
        $status = get_option('gmb_ag_license_status', '');
        $license_key = get_option('gmb_ag_license_key', '');
        
        // Pas de licence du tout
        if (empty($license_key)) {
            return __('Licence non activee. Activez votre licence pour utiliser le plugin.', 'gmb-article-generateur');
        }
        
        switch ($status) {
            case 'blocked':
                return __('Votre licence a ete bloquee. Contactez le support.', 'gmb-article-generateur');
            case 'expired':
                return __('Votre licence a expire. Veuillez la renouveler.', 'gmb-article-generateur');
            case 'invalid':
                return __('Licence invalide. Verifiez votre cle de licence.', 'gmb-article-generateur');
            default:
                return __('Licence non activee. Activez votre licence pour utiliser le plugin.', 'gmb-article-generateur');
        }
    }
}
