<?php
/**
 * Classe de gestion de l'authentification API
 * 
 * @package GMB_Article_Generateur
 * @since 1.0.0
 */

if (!defined('ABSPATH')) {
    exit;
}

class GMB_AG_Auth {

    public static function authenticate($request) {
        $auth_header = $request->get_header('Authorization');

        if (empty($auth_header)) {
            return new WP_Error('gmb_missing_auth', __('Header Authorization manquant.', 'gmb-article-generateur'), array('status' => 401));
        }

        if (!preg_match('/^Bearer\s+(.+)$/i', $auth_header, $matches)) {
            return new WP_Error('gmb_invalid_auth_format', __('Format invalide. Utilisez: Authorization: Bearer CLE', 'gmb-article-generateur'), array('status' => 401));
        }

        $provided_key = trim($matches[1]);
        $stored_key = get_option('gmb_ag_api_key');

        if (!hash_equals($stored_key, $provided_key)) {
            GMB_AG_Logger::log(array(
                'request_type' => $request->get_method(),
                'endpoint'     => $request->get_route(),
                'status'       => 'auth_failed',
                'message'      => __('Cle API invalide', 'gmb-article-generateur'),
            ));
            return new WP_Error('gmb_invalid_api_key', __('Cle API invalide.', 'gmb-article-generateur'), array('status' => 401));
        }

        $ip_check = self::check_ip_whitelist();
        if (is_wp_error($ip_check)) {
            return $ip_check;
        }

        return true;
    }

    public static function check_ip_whitelist() {
        $whitelist = get_option('gmb_ag_ip_whitelist', '');
        if (empty(trim($whitelist))) {
            return true;
        }

        $client_ip = self::get_client_ip();
        $allowed_ips = array_filter(array_map('trim', explode("\n", $whitelist)));

        foreach ($allowed_ips as $allowed_ip) {
            if (strpos($allowed_ip, '/') !== false) {
                if (self::ip_in_range($client_ip, $allowed_ip)) {
                    return true;
                }
            } elseif ($client_ip === $allowed_ip) {
                return true;
            }
        }

        GMB_AG_Logger::log(array(
            'request_type' => 'BLOCKED',
            'endpoint'     => 'ip_whitelist',
            'status'       => 'ip_blocked',
            'message'      => sprintf(__('IP %s non autorisee', 'gmb-article-generateur'), $client_ip),
        ));

        return new WP_Error('gmb_ip_not_allowed', sprintf(__('IP %s non autorisee.', 'gmb-article-generateur'), $client_ip), array('status' => 403));
    }

    public static function get_client_ip() {
        $ip_keys = array('HTTP_CF_CONNECTING_IP', 'HTTP_X_FORWARDED_FOR', 'HTTP_X_REAL_IP', 'HTTP_CLIENT_IP', 'REMOTE_ADDR');
        foreach ($ip_keys as $key) {
            if (!empty($_SERVER[$key])) {
                $ip = $_SERVER[$key];
                if (strpos($ip, ',') !== false) {
                    $ip = trim(explode(',', $ip)[0]);
                }
                if (filter_var($ip, FILTER_VALIDATE_IP)) {
                    return $ip;
                }
            }
        }
        return '0.0.0.0';
    }

    private static function ip_in_range($ip, $range) {
        list($subnet, $bits) = explode('/', $range);
        if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4)) {
            $ip_long = ip2long($ip);
            $subnet_long = ip2long($subnet);
            $mask = -1 << (32 - (int) $bits);
            $subnet_long &= $mask;
            return ($ip_long & $mask) === $subnet_long;
        }
        return false;
    }

    public static function regenerate_api_key() {
        $new_key = 'gmb_' . bin2hex(random_bytes(32));
        update_option('gmb_ag_api_key', $new_key);
        GMB_AG_Logger::log(array(
            'request_type' => 'ADMIN',
            'endpoint'     => 'regenerate_key',
            'status'       => 'success',
            'message'      => __('Cle API regeneree', 'gmb-article-generateur'),
        ));
        return $new_key;
    }
}
