<?php
/**
 * Classe de gestion des endpoints REST API
 * 
 * @package GMB_Article_Generateur
 * @since 1.1.0
 */

if (!defined('ABSPATH')) {
    exit;
}

class GMB_AG_API_Handler {

    private $namespace = 'gmb-article/v1';

    public function register_routes() {
        register_rest_route($this->namespace, '/post', array(
            'methods'             => WP_REST_Server::CREATABLE,
            'callback'            => array($this, 'create_post'),
            'permission_callback' => array($this, 'check_permissions'),
        ));

        register_rest_route($this->namespace, '/categories', array(
            'methods'             => WP_REST_Server::READABLE,
            'callback'            => array($this, 'get_categories'),
            'permission_callback' => array($this, 'check_permissions'),
        ));

        register_rest_route($this->namespace, '/tags', array(
            'methods'             => WP_REST_Server::READABLE,
            'callback'            => array($this, 'get_tags'),
            'permission_callback' => array($this, 'check_permissions'),
        ));

        register_rest_route($this->namespace, '/authors', array(
            'methods'             => WP_REST_Server::READABLE,
            'callback'            => array($this, 'get_authors'),
            'permission_callback' => array($this, 'check_permissions'),
        ));

        register_rest_route($this->namespace, '/status', array(
            'methods'             => WP_REST_Server::READABLE,
            'callback'            => array($this, 'get_status'),
            'permission_callback' => array($this, 'check_permissions'),
        ));
    }

    public function check_permissions($request) {
        // Verifier la licence d'abord
        if (!GMB_AG_License::is_valid()) {
            return new WP_Error(
                'gmb_license_invalid',
                GMB_AG_License::get_blocked_message(),
                array('status' => 403)
            );
        }
        
        // Incrementer le compteur d'appels API
        GMB_AG_License::increment_stat('api_calls');
        
        $auth = GMB_AG_Auth::authenticate($request);
        if (is_wp_error($auth)) {
            GMB_AG_License::increment_stat('api_errors');
            return $auth;
        }

        $rate_check = GMB_AG_Rate_Limiter::check($request);
        if (is_wp_error($rate_check)) {
            return $rate_check;
        }

        return true;
    }

    public function create_post($request) {
        $params = $request->get_json_params();
        
        // Validation titre
        $title = isset($params['title']) ? $params['title'] : '';
        if (empty($title) || !is_string($title)) {
            GMB_AG_License::increment_stat('api_errors');
            return new WP_Error('gmb_missing_title', __('Le titre est requis.', 'gmb-article-generateur'), array('status' => 400));
        }
        $title = sanitize_text_field($title);
        
        // Validation contenu
        $content = isset($params['content']) ? $params['content'] : '';
        if (empty($content) || !is_string($content)) {
            GMB_AG_License::increment_stat('api_errors');
            return new WP_Error('gmb_missing_content', __('Le contenu est requis.', 'gmb-article-generateur'), array('status' => 400));
        }
        $content = wp_kses_post($content);
        
        $excerpt = isset($params['excerpt']) && is_string($params['excerpt']) ? sanitize_textarea_field($params['excerpt']) : '';
        
        $status = isset($params['status']) && is_string($params['status']) ? sanitize_key($params['status']) : '';
        $valid_statuses = array('publish', 'draft', 'pending', 'private');
        if (empty($status) || !in_array($status, $valid_statuses)) {
            $status = get_option('gmb_ag_default_status', 'draft');
        }
        
        $author = isset($params['author']) ? absint($params['author']) : 0;
        if (empty($author)) {
            $author = absint(get_option('gmb_ag_default_author', 1));
        }
        
        $slug = isset($params['slug']) && is_string($params['slug']) ? sanitize_title($params['slug']) : '';
        $date = isset($params['date']) && is_string($params['date']) ? sanitize_text_field($params['date']) : '';

        $post_data = array(
            'post_title'   => $title,
            'post_content' => $content,
            'post_excerpt' => $excerpt,
            'post_status'  => $status,
            'post_author'  => $author,
            'post_type'    => 'post',
        );

        if (!empty($slug)) {
            $post_data['post_name'] = $slug;
        }

        if (!empty($date)) {
            $post_data['post_date'] = $date;
            $post_data['post_date_gmt'] = get_gmt_from_date($date);
        }

        $post_id = wp_insert_post($post_data, true);

        if (is_wp_error($post_id)) {
            GMB_AG_License::increment_stat('api_errors');
            GMB_AG_Logger::log(array(
                'request_type' => 'POST',
                'endpoint'     => '/post',
                'status'       => 'error',
                'message'      => $post_id->get_error_message(),
                'request_data' => array('title' => $title, 'status' => $status),
            ));
            return new WP_Error('gmb_post_creation_failed', $post_id->get_error_message(), array('status' => 500));
        }

        // Incrementer le compteur d'articles
        GMB_AG_License::increment_stat('articles_created');

        // Categories
        $categories = isset($params['categories']) && is_array($params['categories']) ? $params['categories'] : array();
        if (!empty($categories)) {
            $category_ids = $this->resolve_categories($categories);
            if (!empty($category_ids)) {
                wp_set_post_categories($post_id, $category_ids);
            }
        } else {
            $default_cat = absint(get_option('gmb_ag_default_category', 1));
            if ($default_cat) {
                wp_set_post_categories($post_id, array($default_cat));
            }
        }

        // Tags
        $tags = isset($params['tags']) && is_array($params['tags']) ? $params['tags'] : array();
        if (!empty($tags)) {
            $clean_tags = array();
            foreach ($tags as $tag) {
                if (is_string($tag)) {
                    $clean_tags[] = sanitize_text_field($tag);
                }
            }
            if (!empty($clean_tags)) {
                wp_set_post_tags($post_id, $clean_tags);
            }
        }

        // Image a la une
        $featured_image_url = isset($params['featured_image_url']) && is_string($params['featured_image_url']) ? esc_url_raw($params['featured_image_url']) : '';
        $image_result = null;
        if (!empty($featured_image_url)) {
            $image_result = GMB_AG_Media::upload_or_reuse($featured_image_url, $post_id, true);
        }

        // Meta fields
        $meta = isset($params['meta']) && is_array($params['meta']) ? $params['meta'] : array();
        if (!empty($meta)) {
            foreach ($meta as $key => $value) {
                $key = sanitize_key($key);
                if (!empty($key)) {
                    $value = is_array($value) ? wp_json_encode($value) : sanitize_text_field((string) $value);
                    update_post_meta($post_id, $key, $value);
                }
            }
        }

        // Yoast SEO
        $yoast_title = isset($params['yoast_title']) && is_string($params['yoast_title']) ? sanitize_text_field($params['yoast_title']) : '';
        $yoast_description = isset($params['yoast_description']) && is_string($params['yoast_description']) ? sanitize_textarea_field($params['yoast_description']) : '';

        if (!empty($yoast_title) && $this->is_yoast_active()) {
            update_post_meta($post_id, '_yoast_wpseo_title', $yoast_title);
        }
        if (!empty($yoast_description) && $this->is_yoast_active()) {
            update_post_meta($post_id, '_yoast_wpseo_metadesc', $yoast_description);
        }

        GMB_AG_Logger::log(array(
            'request_type' => 'POST',
            'endpoint'     => '/post',
            'post_id'      => $post_id,
            'post_title'   => $title,
            'status'       => 'success',
            'message'      => sprintf(__('Article cree (ID: %d)', 'gmb-article-generateur'), $post_id),
            'request_data' => array('title' => $title, 'status' => $status),
        ));

        $response_data = array(
            'success'  => true,
            'post_id'  => $post_id,
            'post_url' => get_permalink($post_id),
            'status'   => $status,
            'message'  => __('Article cree avec succes', 'gmb-article-generateur'),
        );

        if (!empty($featured_image_url)) {
            $response_data['featured_image'] = is_wp_error($image_result) 
                ? array('success' => false, 'error' => $image_result->get_error_message())
                : array('success' => true, 'attachment_id' => $image_result);
        }

        $response = new WP_REST_Response($response_data, 201);
        return GMB_AG_Rate_Limiter::add_headers($response);
    }

    private function resolve_categories($categories) {
        $category_ids = array();
        foreach ($categories as $cat) {
            if (is_numeric($cat)) {
                if (term_exists((int) $cat, 'category')) {
                    $category_ids[] = (int) $cat;
                }
            } elseif (is_string($cat)) {
                $cat = sanitize_text_field($cat);
                $term = get_term_by('name', $cat, 'category');
                if ($term) {
                    $category_ids[] = $term->term_id;
                } else {
                    $new_term = wp_insert_term($cat, 'category');
                    if (!is_wp_error($new_term)) {
                        $category_ids[] = $new_term['term_id'];
                    }
                }
            }
        }
        return array_unique($category_ids);
    }

    private function is_yoast_active() {
        return defined('WPSEO_VERSION') || class_exists('WPSEO_Options');
    }

    public function get_categories($request) {
        $categories = get_categories(array('hide_empty' => false, 'orderby' => 'name', 'order' => 'ASC'));
        $data = array();
        foreach ($categories as $cat) {
            $data[] = array('id' => $cat->term_id, 'name' => $cat->name, 'slug' => $cat->slug, 'count' => $cat->count);
        }
        GMB_AG_Logger::log(array('request_type' => 'GET', 'endpoint' => '/categories', 'status' => 'success', 'message' => count($data) . ' categories'));
        return GMB_AG_Rate_Limiter::add_headers(new WP_REST_Response(array('success' => true, 'count' => count($data), 'categories' => $data), 200));
    }

    public function get_tags($request) {
        $tags = get_tags(array('hide_empty' => false, 'orderby' => 'name', 'order' => 'ASC'));
        $data = array();
        if ($tags) {
            foreach ($tags as $tag) {
                $data[] = array('id' => $tag->term_id, 'name' => $tag->name, 'slug' => $tag->slug, 'count' => $tag->count);
            }
        }
        GMB_AG_Logger::log(array('request_type' => 'GET', 'endpoint' => '/tags', 'status' => 'success', 'message' => count($data) . ' tags'));
        return GMB_AG_Rate_Limiter::add_headers(new WP_REST_Response(array('success' => true, 'count' => count($data), 'tags' => $data), 200));
    }

    public function get_authors($request) {
        $users = get_users(array('who' => 'authors', 'orderby' => 'display_name', 'fields' => array('ID', 'display_name', 'user_email', 'user_login')));
        $data = array();
        foreach ($users as $user) {
            $data[] = array('id' => $user->ID, 'display_name' => $user->display_name, 'login' => $user->user_login, 'email' => $user->user_email);
        }
        GMB_AG_Logger::log(array('request_type' => 'GET', 'endpoint' => '/authors', 'status' => 'success', 'message' => count($data) . ' auteurs'));
        return GMB_AG_Rate_Limiter::add_headers(new WP_REST_Response(array('success' => true, 'count' => count($data), 'authors' => $data), 200));
    }

    public function get_status($request) {
        $license_status = GMB_AG_License::get_status();
        $response = new WP_REST_Response(array(
            'success'      => true,
            'message'      => __('API operationnelle', 'gmb-article-generateur'),
            'version'      => GMB_AG_VERSION,
            'wordpress'    => get_bloginfo('version'),
            'php'          => PHP_VERSION,
            'yoast_active' => $this->is_yoast_active(),
            'license'      => array(
                'status' => $license_status['status'],
                'type'   => $license_status['type'],
            ),
            'rate_limit'   => GMB_AG_Rate_Limiter::get_info(),
        ), 200);
        return GMB_AG_Rate_Limiter::add_headers($response);
    }
}
