<?php
/**
 * Classe de gestion de l'administration
 * 
 * @package GMB_Article_Generateur
 * @since 1.1.0
 */

if (!defined('ABSPATH')) {
    exit;
}

class GMB_AG_Admin {

    public static function add_admin_menu() {
        add_menu_page(__('Article Generateur', 'gmb-article-generateur'), __('GMB Article', 'gmb-article-generateur'), 'manage_options', 'gmb-article-generateur', array(__CLASS__, 'render_settings_page'), 'dashicons-rest-api', 80);
        add_submenu_page('gmb-article-generateur', __('Reglages', 'gmb-article-generateur'), __('Reglages', 'gmb-article-generateur'), 'manage_options', 'gmb-article-generateur', array(__CLASS__, 'render_settings_page'));
        add_submenu_page('gmb-article-generateur', __('Licence', 'gmb-article-generateur'), __('Licence', 'gmb-article-generateur'), 'manage_options', 'gmb-article-generateur-license', array(__CLASS__, 'render_license_page'));
        add_submenu_page('gmb-article-generateur', __('Logs', 'gmb-article-generateur'), __('Logs', 'gmb-article-generateur'), 'manage_options', 'gmb-article-generateur-logs', array(__CLASS__, 'render_logs_page'));
        add_submenu_page('gmb-article-generateur', __('Documentation', 'gmb-article-generateur'), __('Documentation', 'gmb-article-generateur'), 'manage_options', 'gmb-article-generateur-docs', array(__CLASS__, 'render_docs_page'));
    }

    public static function register_settings() {
        add_action('wp_ajax_gmb_ag_regenerate_key', array(__CLASS__, 'ajax_regenerate_key'));
        add_action('wp_ajax_gmb_ag_clear_logs', array(__CLASS__, 'ajax_clear_logs'));
        add_action('wp_ajax_gmb_ag_export_logs', array(__CLASS__, 'ajax_export_logs'));
        add_action('wp_ajax_gmb_ag_activate_license', array(__CLASS__, 'ajax_activate_license'));
        add_action('wp_ajax_gmb_ag_deactivate_license', array(__CLASS__, 'ajax_deactivate_license'));
        add_action('wp_ajax_gmb_ag_check_license', array(__CLASS__, 'ajax_check_license'));
    }

    public static function render_license_page() {
        if (!current_user_can('manage_options')) {
            wp_die(__('Permission refusee.', 'gmb-article-generateur'));
        }

        $license = GMB_AG_License::get_status();
        $status_labels = array(
            'active' => array('label' => 'Active', 'class' => 'success'),
            'trial' => array('label' => 'Essai', 'class' => 'warning'),
            'expired' => array('label' => 'Expiree', 'class' => 'error'),
            'blocked' => array('label' => 'Bloquee', 'class' => 'error'),
            'invalid' => array('label' => 'Invalide', 'class' => 'error'),
            'unknown' => array('label' => 'Non activee', 'class' => 'warning'),
        );
        $status_info = $status_labels[$license['status']] ?? $status_labels['unknown'];
        ?>
        <div class="wrap gmb-ag-admin">
            <h1><span class="dashicons dashicons-admin-network"></span> <?php _e('Licence GMB Article Generateur', 'gmb-article-generateur'); ?></h1>
            
            <div class="gmb-ag-license-status-card gmb-ag-card">
                <h2><?php _e('Statut de la licence', 'gmb-article-generateur'); ?></h2>
                <div class="gmb-ag-license-info">
                    <div class="gmb-ag-license-badge gmb-ag-badge-<?php echo esc_attr($status_info['class']); ?>">
                        <?php echo esc_html($status_info['label']); ?>
                    </div>
                    <?php if (!empty($license['type'])) : ?>
                        <p><strong><?php _e('Type:', 'gmb-article-generateur'); ?></strong> <?php echo esc_html(ucfirst($license['type'])); ?></p>
                    <?php endif; ?>
                    <?php if (!empty($license['expires_at'])) : ?>
                        <p><strong><?php _e('Expiration:', 'gmb-article-generateur'); ?></strong> <?php echo esc_html(date_i18n(get_option('date_format'), strtotime($license['expires_at']))); ?></p>
                    <?php endif; ?>
                    <?php if (!empty($license['last_check'])) : ?>
                        <p><strong><?php _e('Derniere verification:', 'gmb-article-generateur'); ?></strong> <?php echo esc_html(human_time_diff($license['last_check'])) . ' ' . __('ago', 'gmb-article-generateur'); ?></p>
                    <?php endif; ?>
                </div>
                
                <?php if (in_array($license['status'], array('active', 'trial'))) : ?>
                    <div class="gmb-ag-license-actions">
                        <button type="button" id="gmb_ag_check_license" class="button">
                            <span class="dashicons dashicons-update"></span> <?php _e('Verifier', 'gmb-article-generateur'); ?>
                        </button>
                        <button type="button" id="gmb_ag_deactivate_license" class="button button-secondary">
                            <span class="dashicons dashicons-no"></span> <?php _e('Desactiver', 'gmb-article-generateur'); ?>
                        </button>
                    </div>
                <?php endif; ?>
            </div>
            
            <div class="gmb-ag-card">
                <h2><?php _e('Activer une licence', 'gmb-article-generateur'); ?></h2>
                <p><?php _e('Entrez votre cle de licence pour activer le plugin sur ce site.', 'gmb-article-generateur'); ?></p>
                <div class="gmb-ag-license-form">
                    <input type="text" id="gmb_ag_license_key_input" value="<?php echo esc_attr($license['license_key']); ?>" placeholder="gmb_lic_XXXXXXXXXXXX" class="large-text code">
                    <button type="button" id="gmb_ag_activate_license" class="button button-primary button-hero">
                        <span class="dashicons dashicons-yes"></span> <?php _e('Activer la licence', 'gmb-article-generateur'); ?>
                    </button>
                </div>
            </div>
            
            <div class="gmb-ag-card">
                <h2><?php _e('Besoin d\'une licence ?', 'gmb-article-generateur'); ?></h2>
                <p><?php _e('Contactez GMB Club pour obtenir votre licence.', 'gmb-article-generateur'); ?></p>
                <a href="https://avismaestrogmb.com/contact" target="_blank" class="button"><?php _e('Nous contacter', 'gmb-article-generateur'); ?></a>
            </div>
            
            <div class="gmb-ag-footer">
                <p><?php _e('Developpe par', 'gmb-article-generateur'); ?> <a href="https://avismaestrogmb.com" target="_blank">GMB Club</a></p>
            </div>
        </div>
        <?php
    }

    public static function render_settings_page() {
        if (!current_user_can('manage_options')) {
            wp_die(__('Permission refusee.', 'gmb-article-generateur'));
        }

        // Verifier si licence active
        $license = GMB_AG_License::get_status();
        $license_valid = in_array($license['status'], array('active', 'trial'));
        
        // Afficher alerte si licence non activee
        if (!$license_valid) {
            ?>
            <div class="wrap gmb-ag-admin">
                <h1><span class="dashicons dashicons-rest-api"></span> <?php _e('GMB Club - Article Generateur SEO', 'gmb-article-generateur'); ?></h1>
                <div class="gmb-ag-license-required">
                    <div class="gmb-ag-license-popup">
                        <span class="dashicons dashicons-lock" style="font-size:48px;width:48px;height:48px;color:#dc3545;"></span>
                        <h2><?php _e('Licence requise', 'gmb-article-generateur'); ?></h2>
                        <p><?php _e('Vous devez activer votre licence pour utiliser ce plugin.', 'gmb-article-generateur'); ?></p>
                        <p><?php _e('L\'API REST est desactivee tant que la licence n\'est pas valide.', 'gmb-article-generateur'); ?></p>
                        <a href="<?php echo admin_url('admin.php?page=gmb-article-generateur-license'); ?>" class="button button-primary button-hero">
                            <span class="dashicons dashicons-admin-network"></span> <?php _e('Activer ma licence', 'gmb-article-generateur'); ?>
                        </a>
                    </div>
                </div>
            </div>
            <style>
            .gmb-ag-license-required{display:flex;justify-content:center;align-items:center;min-height:400px}
            .gmb-ag-license-popup{background:#fff;border:2px solid #dc3545;border-radius:10px;padding:40px;text-align:center;max-width:500px;box-shadow:0 4px 20px rgba(220,53,69,.2)}
            .gmb-ag-license-popup h2{margin:20px 0 10px;color:#dc3545}
            .gmb-ag-license-popup p{color:#666;margin:10px 0}
            .gmb-ag-license-popup .button-hero{margin-top:20px;display:inline-flex;align-items:center;gap:8px}
            </style>
            <?php
            return;
        }

        if (isset($_POST['gmb_ag_save_settings']) && check_admin_referer('gmb_ag_settings_nonce')) {
            update_option('gmb_ag_default_status', sanitize_key($_POST['gmb_ag_default_status'] ?? 'draft'));
            update_option('gmb_ag_default_author', absint($_POST['gmb_ag_default_author'] ?? 1));
            update_option('gmb_ag_default_category', absint($_POST['gmb_ag_default_category'] ?? 1));
            update_option('gmb_ag_ip_whitelist', sanitize_textarea_field($_POST['gmb_ag_ip_whitelist'] ?? ''));
            update_option('gmb_ag_rate_limit', absint($_POST['gmb_ag_rate_limit'] ?? 60));
            update_option('gmb_ag_enable_logging', isset($_POST['gmb_ag_enable_logging']) ? 1 : 0);
            echo '<div class="notice notice-success is-dismissible"><p>' . __('Reglages enregistres.', 'gmb-article-generateur') . '</p></div>';
        }

        $api_key = get_option('gmb_ag_api_key', '');
        $default_status = get_option('gmb_ag_default_status', 'draft');
        $default_author = get_option('gmb_ag_default_author', 1);
        $default_category = get_option('gmb_ag_default_category', 1);
        $ip_whitelist = get_option('gmb_ag_ip_whitelist', '');
        $rate_limit = get_option('gmb_ag_rate_limit', 60);
        $enable_logging = get_option('gmb_ag_enable_logging', 1);
        $authors = get_users(array('who' => 'authors'));
        $categories = get_categories(array('hide_empty' => false));
        $api_base_url = rest_url('gmb-article/v1/');
        $stats = GMB_AG_Logger::get_stats();
        ?>
        <div class="wrap gmb-ag-admin">
            <h1><span class="dashicons dashicons-rest-api"></span> <?php _e('GMB Club - Article Generateur SEO', 'gmb-article-generateur'); ?></h1>
            <div class="gmb-ag-header-info">
                <p><?php _e('Publiez des articles WordPress via une API REST securisee.', 'gmb-article-generateur'); ?></p>
                <p><strong><?php _e('URL de l\'API :', 'gmb-article-generateur'); ?></strong> <code><?php echo esc_html($api_base_url); ?></code></p>
            </div>
            <form method="post">
                <?php wp_nonce_field('gmb_ag_settings_nonce'); ?>
                <div class="gmb-ag-settings-grid">
                    <div class="gmb-ag-card">
                        <h2><?php _e('Cle API', 'gmb-article-generateur'); ?></h2>
                        <div class="gmb-ag-api-key-container">
                            <input type="text" id="gmb_ag_api_key" value="<?php echo esc_attr($api_key); ?>" readonly class="large-text code">
                            <div class="gmb-ag-api-key-actions">
                                <button type="button" id="gmb_ag_copy_key" class="button"><span class="dashicons dashicons-clipboard"></span> <?php _e('Copier', 'gmb-article-generateur'); ?></button>
                                <button type="button" id="gmb_ag_regenerate_key" class="button button-secondary"><span class="dashicons dashicons-update"></span> <?php _e('Regenerer', 'gmb-article-generateur'); ?></button>
                            </div>
                        </div>
                    </div>
                    <div class="gmb-ag-card">
                        <h2><?php _e('Parametres par defaut', 'gmb-article-generateur'); ?></h2>
                        <table class="form-table">
                            <tr><th><label for="gmb_ag_default_status"><?php _e('Statut', 'gmb-article-generateur'); ?></label></th>
                                <td><select name="gmb_ag_default_status" id="gmb_ag_default_status">
                                    <option value="draft" <?php selected($default_status, 'draft'); ?>><?php _e('Brouillon', 'gmb-article-generateur'); ?></option>
                                    <option value="publish" <?php selected($default_status, 'publish'); ?>><?php _e('Publie', 'gmb-article-generateur'); ?></option>
                                    <option value="pending" <?php selected($default_status, 'pending'); ?>><?php _e('En attente', 'gmb-article-generateur'); ?></option>
                                </select></td></tr>
                            <tr><th><label for="gmb_ag_default_author"><?php _e('Auteur', 'gmb-article-generateur'); ?></label></th>
                                <td><select name="gmb_ag_default_author" id="gmb_ag_default_author">
                                    <?php foreach ($authors as $author) : ?>
                                        <option value="<?php echo esc_attr($author->ID); ?>" <?php selected($default_author, $author->ID); ?>><?php echo esc_html($author->display_name); ?></option>
                                    <?php endforeach; ?>
                                </select></td></tr>
                            <tr><th><label for="gmb_ag_default_category"><?php _e('Categorie', 'gmb-article-generateur'); ?></label></th>
                                <td><select name="gmb_ag_default_category" id="gmb_ag_default_category">
                                    <?php foreach ($categories as $cat) : ?>
                                        <option value="<?php echo esc_attr($cat->term_id); ?>" <?php selected($default_category, $cat->term_id); ?>><?php echo esc_html($cat->name); ?></option>
                                    <?php endforeach; ?>
                                </select></td></tr>
                        </table>
                    </div>
                    <div class="gmb-ag-card">
                        <h2><?php _e('Securite', 'gmb-article-generateur'); ?></h2>
                        <table class="form-table">
                            <tr><th><label for="gmb_ag_rate_limit"><?php _e('Limite requetes/heure', 'gmb-article-generateur'); ?></label></th>
                                <td><input type="number" name="gmb_ag_rate_limit" id="gmb_ag_rate_limit" value="<?php echo esc_attr($rate_limit); ?>" min="0" max="1000" class="small-text"></td></tr>
                            <tr><th><label for="gmb_ag_ip_whitelist"><?php _e('Whitelist IP', 'gmb-article-generateur'); ?></label></th>
                                <td><textarea name="gmb_ag_ip_whitelist" id="gmb_ag_ip_whitelist" rows="3" class="large-text code"><?php echo esc_textarea($ip_whitelist); ?></textarea></td></tr>
                            <tr><th><label for="gmb_ag_enable_logging"><?php _e('Logs', 'gmb-article-generateur'); ?></label></th>
                                <td><label><input type="checkbox" name="gmb_ag_enable_logging" value="1" <?php checked($enable_logging, 1); ?>> <?php _e('Activer', 'gmb-article-generateur'); ?></label></td></tr>
                        </table>
                    </div>
                </div>
                <p class="submit"><input type="submit" name="gmb_ag_save_settings" class="button button-primary button-hero" value="<?php _e('Enregistrer', 'gmb-article-generateur'); ?>"></p>
            </form>
            <div class="gmb-ag-card gmb-ag-stats">
                <h2><?php _e('Statistiques', 'gmb-article-generateur'); ?></h2>
                <div class="gmb-ag-stats-grid">
                    <div class="gmb-ag-stat"><span class="gmb-ag-stat-value"><?php echo number_format_i18n($stats['total']); ?></span><span class="gmb-ag-stat-label"><?php _e('Total requetes', 'gmb-article-generateur'); ?></span></div>
                    <div class="gmb-ag-stat"><span class="gmb-ag-stat-value"><?php echo number_format_i18n($stats['last_24h']); ?></span><span class="gmb-ag-stat-label"><?php _e('Dernieres 24h', 'gmb-article-generateur'); ?></span></div>
                    <div class="gmb-ag-stat"><span class="gmb-ag-stat-value"><?php echo number_format_i18n($stats['unique_ips']); ?></span><span class="gmb-ag-stat-label"><?php _e('IPs uniques', 'gmb-article-generateur'); ?></span></div>
                </div>
            </div>
            <div class="gmb-ag-footer"><p><?php _e('Developpe par', 'gmb-article-generateur'); ?> <a href="https://avismaestrogmb.com" target="_blank">GMB Club</a></p></div>
        </div>
        <?php
    }

    public static function render_logs_page() {
        if (!current_user_can('manage_options')) { wp_die(__('Permission refusee.', 'gmb-article-generateur')); }
        $page = isset($_GET['paged']) ? absint($_GET['paged']) : 1;
        $status = isset($_GET['status']) ? sanitize_key($_GET['status']) : '';
        $search = isset($_GET['s']) ? sanitize_text_field($_GET['s']) : '';
        $result = GMB_AG_Logger::get_logs(array('page' => $page, 'per_page' => 20, 'status' => $status, 'search' => $search));
        ?>
        <div class="wrap gmb-ag-admin">
            <h1><span class="dashicons dashicons-list-view"></span> <?php _e('Logs des requetes API', 'gmb-article-generateur'); ?></h1>
            <div class="gmb-ag-logs-actions">
                <form method="get" class="gmb-ag-search-form">
                    <input type="hidden" name="page" value="gmb-article-generateur-logs">
                    <input type="search" name="s" value="<?php echo esc_attr($search); ?>" placeholder="<?php _e('Rechercher...', 'gmb-article-generateur'); ?>">
                    <select name="status"><option value=""><?php _e('Tous', 'gmb-article-generateur'); ?></option><option value="success" <?php selected($status, 'success'); ?>>Succes</option><option value="error" <?php selected($status, 'error'); ?>>Erreur</option></select>
                    <button type="submit" class="button"><?php _e('Filtrer', 'gmb-article-generateur'); ?></button>
                </form>
                <div class="gmb-ag-logs-bulk-actions">
                    <button type="button" id="gmb_ag_export_logs" class="button"><span class="dashicons dashicons-download"></span> <?php _e('Exporter', 'gmb-article-generateur'); ?></button>
                    <button type="button" id="gmb_ag_clear_logs" class="button button-secondary"><span class="dashicons dashicons-trash"></span> <?php _e('Vider', 'gmb-article-generateur'); ?></button>
                </div>
            </div>
            <table class="wp-list-table widefat fixed striped">
                <thead><tr><th><?php _e('Date', 'gmb-article-generateur'); ?></th><th><?php _e('Endpoint', 'gmb-article-generateur'); ?></th><th><?php _e('IP', 'gmb-article-generateur'); ?></th><th><?php _e('Article', 'gmb-article-generateur'); ?></th><th><?php _e('Statut', 'gmb-article-generateur'); ?></th><th><?php _e('Message', 'gmb-article-generateur'); ?></th></tr></thead>
                <tbody>
                <?php if (empty($result['logs'])) : ?>
                    <tr><td colspan="6"><?php _e('Aucun log.', 'gmb-article-generateur'); ?></td></tr>
                <?php else : foreach ($result['logs'] as $log) : ?>
                    <tr>
                        <td><?php echo esc_html(wp_date('d/m H:i', strtotime($log['date_created']))); ?></td>
                        <td><code><?php echo esc_html($log['endpoint']); ?></code></td>
                        <td><code><?php echo esc_html($log['ip_address']); ?></code></td>
                        <td><?php echo $log['post_id'] ? '<a href="' . get_edit_post_link($log['post_id']) . '">#' . $log['post_id'] . '</a>' : '-'; ?></td>
                        <td><span class="gmb-ag-status gmb-ag-status-<?php echo esc_attr($log['status']); ?>"><?php echo esc_html($log['status']); ?></span></td>
                        <td><?php echo esc_html($log['message']); ?></td>
                    </tr>
                <?php endforeach; endif; ?>
                </tbody>
            </table>
            <?php if ($result['pages'] > 1) : ?>
                <div class="tablenav bottom"><div class="tablenav-pages"><?php echo paginate_links(array('base' => add_query_arg('paged', '%#%'), 'total' => $result['pages'], 'current' => $page)); ?></div></div>
            <?php endif; ?>
        </div>
        <?php
    }

    public static function render_docs_page() {
        $api_base_url = rest_url('gmb-article/v1/');
        $api_key = get_option('gmb_ag_api_key', 'VOTRE_CLE_API');
        ?>
        <div class="wrap gmb-ag-admin gmb-ag-docs">
            <h1><span class="dashicons dashicons-book"></span> <?php _e('Documentation', 'gmb-article-generateur'); ?></h1>
            <div class="gmb-ag-card">
                <h2><?php _e('Exemple curl', 'gmb-article-generateur'); ?></h2>
                <pre><code>curl -X POST "<?php echo esc_html($api_base_url); ?>post" -H "Authorization: Bearer <?php echo esc_html($api_key); ?>" -H "Content-Type: application/json" -d '{"title": "Mon article", "content": "&lt;p&gt;Contenu&lt;/p&gt;", "status": "draft"}'</code></pre>
            </div>
            <div class="gmb-ag-card">
                <h2><?php _e('Parametres', 'gmb-article-generateur'); ?></h2>
                <table class="wp-list-table widefat"><thead><tr><th>Parametre</th><th>Requis</th><th>Description</th></tr></thead><tbody>
                    <tr><td><code>title</code></td><td>Oui</td><td>Titre</td></tr>
                    <tr><td><code>content</code></td><td>Oui</td><td>Contenu HTML</td></tr>
                    <tr><td><code>status</code></td><td></td><td>publish, draft, pending</td></tr>
                    <tr><td><code>categories</code></td><td></td><td>IDs ou noms</td></tr>
                    <tr><td><code>tags</code></td><td></td><td>Noms</td></tr>
                    <tr><td><code>featured_image_url</code></td><td></td><td>URL image</td></tr>
                    <tr><td><code>yoast_title</code></td><td></td><td>Titre SEO</td></tr>
                    <tr><td><code>yoast_description</code></td><td></td><td>Meta description</td></tr>
                </tbody></table>
            </div>
        </div>
        <?php
    }

    // AJAX handlers
    public static function ajax_regenerate_key() {
        check_ajax_referer('gmb_ag_admin_nonce', 'nonce');
        if (!current_user_can('manage_options')) wp_send_json_error('Permission refusee');
        wp_send_json_success(array('key' => GMB_AG_Auth::regenerate_api_key()));
    }

    public static function ajax_clear_logs() {
        check_ajax_referer('gmb_ag_admin_nonce', 'nonce');
        if (!current_user_can('manage_options')) wp_send_json_error('Permission refusee');
        GMB_AG_Logger::clear_all();
        wp_send_json_success(array('message' => 'Logs supprimes'));
    }

    public static function ajax_export_logs() {
        check_ajax_referer('gmb_ag_admin_nonce', 'nonce');
        if (!current_user_can('manage_options')) wp_send_json_error('Permission refusee');
        wp_send_json_success(array('csv' => base64_encode(GMB_AG_Logger::export_csv()), 'filename' => 'gmb-logs-' . date('Y-m-d') . '.csv'));
    }

    public static function ajax_activate_license() {
        check_ajax_referer('gmb_ag_admin_nonce', 'nonce');
        if (!current_user_can('manage_options')) wp_send_json_error('Permission refusee');
        
        $license_key = isset($_POST['license_key']) ? sanitize_text_field($_POST['license_key']) : '';
        if (empty($license_key)) {
            wp_send_json_error('Cle de licence requise');
        }
        
        $result = GMB_AG_License::activate($license_key);
        
        if (!empty($result['valid'])) {
            wp_send_json_success($result);
        } else {
            wp_send_json_error($result['message'] ?? 'Echec de l\'activation');
        }
    }

    public static function ajax_deactivate_license() {
        check_ajax_referer('gmb_ag_admin_nonce', 'nonce');
        if (!current_user_can('manage_options')) wp_send_json_error('Permission refusee');
        
        $result = GMB_AG_License::deactivate();
        wp_send_json_success(array('message' => 'Licence desactivee'));
    }

    public static function ajax_check_license() {
        check_ajax_referer('gmb_ag_admin_nonce', 'nonce');
        if (!current_user_can('manage_options')) wp_send_json_error('Permission refusee');
        
        $result = GMB_AG_License::check();
        
        if (!empty($result['valid'])) {
            wp_send_json_success($result);
        } else {
            wp_send_json_error($result);
        }
    }
}
