<?php
/**
 * Plugin Name: GMB Club - Article Generateur SEO
 * Plugin URI: https://avismaestrogmb.com/plugins/article-generateur
 * Description: Publiez des articles WordPress via une API REST securisee. Ideal pour l'automatisation avec n8n, Make, Zapier ou scripts Python.
 * Version: 1.1.0
 * Author: GMB Club
 * Author URI: https://avismaestrogmb.com
 * License: GPL v2 or later
 * Text Domain: gmb-article-generateur
 * Requires at least: 6.0
 * Requires PHP: 7.4
 */

if (!defined('ABSPATH')) {
    exit;
}

define('GMB_AG_VERSION', '1.1.0');
define('GMB_AG_PLUGIN_FILE', __FILE__);
define('GMB_AG_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('GMB_AG_PLUGIN_URL', plugin_dir_url(__FILE__));
define('GMB_AG_PLUGIN_BASENAME', plugin_basename(__FILE__));

final class GMB_Article_Generateur {

    private static $instance = null;

    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct() {
        $this->load_dependencies();
        $this->init_hooks();
    }

    private function load_dependencies() {
        require_once GMB_AG_PLUGIN_DIR . 'includes/class-license.php';
        require_once GMB_AG_PLUGIN_DIR . 'includes/class-logger.php';
        require_once GMB_AG_PLUGIN_DIR . 'includes/class-auth.php';
        require_once GMB_AG_PLUGIN_DIR . 'includes/class-rate-limiter.php';
        require_once GMB_AG_PLUGIN_DIR . 'includes/class-media.php';
        require_once GMB_AG_PLUGIN_DIR . 'includes/class-api-handler.php';
        
        if (is_admin()) {
            require_once GMB_AG_PLUGIN_DIR . 'includes/class-admin.php';
        }
    }

    private function init_hooks() {
        register_activation_hook(GMB_AG_PLUGIN_FILE, array($this, 'activate'));
        register_deactivation_hook(GMB_AG_PLUGIN_FILE, array($this, 'deactivate'));

        add_action('init', array($this, 'load_textdomain'));
        add_action('rest_api_init', array($this, 'register_rest_routes'));
        
        // Heartbeat cron
        add_action('gmb_ag_license_heartbeat', array('GMB_AG_License', 'heartbeat'));

        if (is_admin()) {
            add_action('admin_menu', array('GMB_AG_Admin', 'add_admin_menu'));
            add_action('admin_init', array('GMB_AG_Admin', 'register_settings'));
            add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_assets'));
            add_action('admin_notices', array($this, 'license_admin_notice'));
        }

        add_filter('plugin_action_links_' . GMB_AG_PLUGIN_BASENAME, array($this, 'add_settings_link'));
    }

    public function activate() {
        $this->create_logs_table();

        if (!get_option('gmb_ag_api_key')) {
            update_option('gmb_ag_api_key', $this->generate_api_key());
        }

        $default_options = array(
            'default_status'   => 'draft',
            'default_author'   => 1,
            'default_category' => 1,
            'ip_whitelist'     => '',
            'rate_limit'       => 60,
            'enable_logging'   => 1,
        );

        foreach ($default_options as $key => $value) {
            if (false === get_option('gmb_ag_' . $key)) {
                update_option('gmb_ag_' . $key, $value);
            }
        }

        // Planifier le heartbeat
        if (!wp_next_scheduled('gmb_ag_license_heartbeat')) {
            wp_schedule_event(time() + 3600, 'daily', 'gmb_ag_license_heartbeat');
        }

        flush_rewrite_rules();
    }

    public function deactivate() {
        // Notifier le serveur de licence
        GMB_AG_License::deactivate();
        
        // Supprimer le cron
        wp_clear_scheduled_hook('gmb_ag_license_heartbeat');
        
        flush_rewrite_rules();
    }

    private function create_logs_table() {
        global $wpdb;

        $table_name = $wpdb->prefix . 'gmb_ag_logs';
        $charset_collate = $wpdb->get_charset_collate();

        $sql = "CREATE TABLE IF NOT EXISTS $table_name (
            id bigint(20) UNSIGNED NOT NULL AUTO_INCREMENT,
            date_created datetime DEFAULT CURRENT_TIMESTAMP NOT NULL,
            request_type varchar(10) NOT NULL,
            endpoint varchar(100) NOT NULL,
            ip_address varchar(45) NOT NULL,
            post_id bigint(20) UNSIGNED DEFAULT NULL,
            post_title varchar(255) DEFAULT NULL,
            status varchar(20) NOT NULL,
            message text,
            request_data longtext,
            PRIMARY KEY (id),
            KEY date_created (date_created),
            KEY ip_address (ip_address),
            KEY status (status)
        ) $charset_collate;";

        require_once ABSPATH . 'wp-admin/includes/upgrade.php';
        dbDelta($sql);
    }

    public function generate_api_key() {
        return 'gmb_' . bin2hex(random_bytes(32));
    }

    public function load_textdomain() {
        load_plugin_textdomain('gmb-article-generateur', false, dirname(GMB_AG_PLUGIN_BASENAME) . '/languages');
    }

    public function register_rest_routes() {
        $api_handler = new GMB_AG_API_Handler();
        $api_handler->register_routes();
    }

    public function enqueue_admin_assets($hook) {
        if (strpos($hook, 'gmb-article-generateur') === false) {
            return;
        }

        wp_enqueue_style('gmb-ag-admin', GMB_AG_PLUGIN_URL . 'assets/css/admin.css', array(), GMB_AG_VERSION);
        wp_enqueue_script('gmb-ag-admin', GMB_AG_PLUGIN_URL . 'assets/js/admin.js', array('jquery'), GMB_AG_VERSION, true);

        wp_localize_script('gmb-ag-admin', 'gmb_ag_admin', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce'    => wp_create_nonce('gmb_ag_admin_nonce'),
            'strings'  => array(
                'confirm_regenerate' => __('Regenerer la cle API ?', 'gmb-article-generateur'),
                'copied'             => __('Copie !', 'gmb-article-generateur'),
                'confirm_clear_logs' => __('Supprimer tous les logs ?', 'gmb-article-generateur'),
                'activating'         => __('Activation...', 'gmb-article-generateur'),
            ),
        ));
    }

    public function add_settings_link($links) {
        $settings_link = sprintf('<a href="%s">%s</a>', admin_url('admin.php?page=gmb-article-generateur'), __('Reglages', 'gmb-article-generateur'));
        array_unshift($links, $settings_link);
        return $links;
    }
    
    public function license_admin_notice() {
        $license_key = get_option('gmb_ag_license_key', '');
        $status = get_option('gmb_ag_license_status', '');
        
        // Licence non configuree
        if (empty($license_key) || empty($status)) {
            echo '<div class="notice notice-error"><p><strong>GMB Article Generateur:</strong> ' . __('Licence non activee. L\'API est desactivee.', 'gmb-article-generateur') . ' <a href="' . admin_url('admin.php?page=gmb-article-generateur-license') . '" class="button button-small">' . __('Activer maintenant', 'gmb-article-generateur') . '</a></p></div>';
            return;
        }
        
        // Ne pas afficher sur la page du plugin (sauf si probleme)
        $screen = get_current_screen();
        if ($screen && strpos($screen->id, 'gmb-article-generateur') !== false) {
            return;
        }
        
        if ($status === 'blocked') {
            echo '<div class="notice notice-error"><p><strong>GMB Article Generateur:</strong> ' . esc_html(GMB_AG_License::get_blocked_message()) . ' <a href="' . admin_url('admin.php?page=gmb-article-generateur-license') . '">Gerer la licence</a></p></div>';
        } elseif ($status === 'expired') {
            echo '<div class="notice notice-warning"><p><strong>GMB Article Generateur:</strong> ' . esc_html(GMB_AG_License::get_blocked_message()) . ' <a href="' . admin_url('admin.php?page=gmb-article-generateur-license') . '">Renouveler</a></p></div>';
        }
    }
}

function gmb_article_generateur() {
    return GMB_Article_Generateur::get_instance();
}

gmb_article_generateur();
